<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Camp;
use Illuminate\Validation\Rule;
use Log;

class CampController extends Controller
{
    public function create()
    {
        return view('pages.office.camp.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'camp_location' => 'required|string|max:255',
            'camp_date' => 'required|date',
            'pt_name' => 'required|string|max:255',
            'age' => 'nullable|integer|min:0|max:120',
            'gender' => 'nullable|in:Male,Female',
            'phone' => 'nullable|string|max:15',
            'phone_status' => 'nullable|string|max:255',
            'bp' => 'nullable|string|max:10',
            'rbs' => 'nullable|string|max:10',
            'feedback_status' => 'nullable|string|in:Yes,No',
            'remark' => 'nullable|string|max:1000',
        ]);

        $camp = new Camp();
        $camp->camp_location = $request->camp_location;
        $camp->camp_date = $request->camp_date;
        $camp->pt_name = $request->pt_name;
        $camp->age = $request->age;
        $camp->gender = $request->gender;
        $camp->phone = $request->phone;
        $camp->phone_status = $request->phone_status;
        $camp->bp = $request->bp;
        $camp->rbs = $request->rbs;
        $camp->feedback_status = $request->feedback_status;
        $camp->remark = $request->remark;
        $camp->save();
        session(['last_camp_location' => $request->camp_location]);
        session(['last_camp_date' => $request->camp_date]);


        return redirect()->route('camp.create')->with('success', 'Camp details have been added successfully.');
    }

    public function index(Request $request)
{
    $query = Camp::query();

    if ($request->filled('phone')) {
        $query->where('phone', 'like', '%' . $request->phone . '%');
    }

    if ($request->filled('pt_name')) {
        $query->where('pt_name', 'like', '%' . $request->pt_name . '%');
    }

    if ($request->filled('from_date') && $request->filled('to_date')) {
        $query->whereBetween('camp_date', [$request->from_date, $request->to_date]);
    } elseif ($request->filled('from_date')) {
        $query->where('camp_date', '>=', $request->from_date);
    } elseif ($request->filled('to_date')) {
        $query->where('camp_date', '<=', $request->to_date);
    }

    if ($request->filled('phone_status')) {
        $query->where('phone_status', $request->phone_status);
    }

    $camps = $query->paginate(10);

    $totalConnected = $camps->where('phone_status', 'Connected')->count();
    $totalPhoneCut = $camps->where('phone_status', 'Phone Cut')->count();
    $totalSwitchedOff = $camps->where('phone_status', 'Switched Off')->count();
    $totalOutOfReach = $camps->where('phone_status', 'Out of Reach')->count();
    $totalNotPicking = $camps->where('phone_status', 'Not Picking')->count();
    $totalBusy = $camps->where('phone_status', 'Busy')->count();
    $totalNoAnswer = $camps->where('phone_status', 'No Answer')->count();
    $totalCallDropped = $camps->where('phone_status', 'Call Dropped')->count();
    $totalVoicemail = $camps->where('phone_status', 'Voicemail')->count();
    $totalNotInService = $camps->where('phone_status', 'Number Not in Service')->count();

    return view('pages.office.camp.index', compact(
        'camps', 
        'totalConnected', 
        'totalPhoneCut', 
        'totalSwitchedOff', 
        'totalOutOfReach', 
        'totalNotPicking', 
        'totalBusy', 
        'totalNoAnswer', 
        'totalCallDropped', 
        'totalVoicemail', 
        'totalNotInService'
    ));
}


    public function edit($id)
    {
        $camp = Camp::findOrFail($id);
        return view('pages.office.camp.edit', compact('camp'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'camp_location' => 'required|string|max:255',
            'camp_date' => 'required|date',
            'pt_name' => 'required|string|max:255',
            'age' => 'nullable|integer|min:0|max:120',
            'gender' => 'nullable|in:Male,Female',
            'phone' => 'nullable|string|max:15',
            'phone_status' => 'nullable|string|max:255',
            'bp' => 'nullable|string|max:10',
            'rbs' => 'nullable|string|max:10',
            'feedback_status' => 'nullable|string|in:Yes,No',
            'remark' => 'nullable|string|max:1000',
        ]);

        $camp = Camp::findOrFail($id);
        $camp->fill($request->all());
        $camp->save();

        return redirect()->route('camp.index')->with('success', 'Camp details have been updated successfully.');
    }

    public function destroy($id)
    {
        $camp = Camp::findOrFail($id);
        $camp->delete();

        return redirect()->route('camp.index')->with('success', 'Camp details have been deleted successfully.');
    }
}
