<?php

namespace App\Http\Controllers;

use App\Models\Accessory;
use App\Models\Laptop;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AccessoryController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'laptop_id' => 'required|exists:laptops,id',
            'employee_id' => 'nullable|exists:employees,id', // Use employee_id instead of user_id
            'name' => 'required|string|max:255',
            'price' => 'required|numeric',
            'approved_by' => 'nullable|string|max:255',
            'bill_file' => 'nullable|mimes:png,jpg,jpeg,pdf|max:2048'
        ]);

        // Handle file upload if present
        $billPath = null;
        if ($request->hasFile('bill_file')) {
            $billPath = $request->file('bill_file')->store('bills', 'public');
        }

        // Create new accessory record
        Accessory::create([
            'laptop_id' => $request->laptop_id,
            'employee_id' => $request->employee_id, // Storing employee_id
            'name' => $request->name,
            'price' => $request->price,
            'approved_by' => $request->approved_by,
            'bill_file' => $billPath,
        ]);

        return redirect()->route('laptop_tracker.index')->with('success', 'Accessory added successfully.');
    }

    public function edit($id)
{
    $accessory = Accessory::findOrFail($id);
    return view('accessories.edit', compact('accessory'));
}

public function update(Request $request, $id)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'price' => 'required|numeric',
        'approved_by' => 'nullable|string|max:255',
        'bill_file' => 'nullable|mimes:png,jpg,jpeg,pdf|max:2048'
    ]);

    $accessory = Accessory::findOrFail($id);

    // Handle bill file update
    $billPath = $accessory->bill_file; // Keep the old file if no new file is uploaded
    if ($request->hasFile('bill_file')) {
        // Delete the old file if it exists
        if ($accessory->bill_file && Storage::disk('public')->exists($accessory->bill_file)) {
            Storage::disk('public')->delete($accessory->bill_file);
        }
        // Store the new file
        $billPath = $request->file('bill_file')->store('bills', 'public');
    }

    // Update accessory
    $accessory->update([
        'name' => $request->name,
        'price' => $request->price,
        'approved_by' => $request->approved_by,
        'bill_file' => $billPath,
    ]);

    return redirect()->route('laptop_tracker.index')->with('success', 'Accessory updated successfully.');
}

public function destroy($id)
{
    $accessory = Accessory::findOrFail($id);

    // Delete the bill file if it exists
    if ($accessory->bill_file && Storage::disk('public')->exists($accessory->bill_file)) {
        Storage::disk('public')->delete($accessory->bill_file);
    }

    // Delete the accessory
    $accessory->delete();

    return redirect()->route('laptop_tracker.index')->with('success', 'Accessory deleted successfully.');
}


    /**
     * Download the bill file for an accessory.
     */
    public function download($id)
{
    $accessory = Accessory::findOrFail($id);

    // Check if the file exists before downloading
    if ($accessory->bill_file && Storage::disk('public')->exists($accessory->bill_file)) {
        // Fetch employee name or default to a placeholder if employee is not found
        $employeeName = $accessory->employee ? $accessory->employee->name : 'Employee_Not_Found';
        $accessoryName = $accessory->name;

        // Generate a proper download filename using the employee and accessory name
        $fileName = str_replace(' ', '_', $employeeName) . '_' . str_replace(' ', '_', $accessoryName) . '.' . pathinfo($accessory->bill_file, PATHINFO_EXTENSION);
        
        // Return the file download with a custom name
        return Storage::disk('public')->download($accessory->bill_file, $fileName);
    }

    return back()->with('error', 'No bill available to download or file not found.');
}




}
