<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Employee;
use App\Models\Laptop;
use App\Models\LaptopHistory;
use Carbon\Carbon;
use App\Models\LaptopRepair;
use Illuminate\Support\Facades\Storage;
use DB;


class LaptopTrackerController extends Controller
{
    public function index(Request $request)
    {
        $query = Laptop::with(['employee', 'repairs']);
        
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where('serial_number', 'like', "%{$search}%")
                  ->orWhereHas('employee', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
        }
    
        if ($request->has('device_type')) {
            $deviceType = $request->get('device_type');
            $query->where('device_type', $deviceType);
        }
    
        $laptops = $query->paginate(10);
        $totalLaptops = Laptop::count();
        $totalDellLaptops = Laptop::where('company_name', 'Dell')->count();
        $totalLenovoLaptops = Laptop::where('company_name', 'Lenovo')->count();
        
        $employeesWithoutLaptop = Employee::doesntHave('laptop')->count();
    
        $employees = Employee::orderBy('id', 'desc')->get();
        $deviceCounts = DB::table('laptops')
        ->join('employees', 'laptops.employee_id', '=', 'employees.id')
        ->select('employees.department', DB::raw('SUM(CASE WHEN device_type = "laptop" THEN 1 ELSE 0 END) as laptop_count'), DB::raw('SUM(CASE WHEN device_type = "desktop" THEN 1 ELSE 0 END) as desktop_count'))
        ->groupBy('employees.department')
        ->get();
    
        return view('pages.laptop_tracker.index', compact('laptops', 'employees', 'totalLaptops', 'totalDellLaptops', 'totalLenovoLaptops', 'employeesWithoutLaptop','deviceCounts'));
    }
    

public function create(Request $request)
{
    $employees = Employee::orderBy('id', 'desc')->get();
    $selectedEmployee = null;
    if ($request->has('employee_id')) {
        $selectedEmployee = Employee::find($request->employee_id);
    }
    return view('pages.laptop_tracker.create', compact('employees', 'selectedEmployee'));
}


     
 

public function store(Request $request)
{
    $request->validate([
        'employee_id' => 'required|exists:employees,id',
        'phone' => 'required|string|max:15',
        'emergency_contact' => 'nullable|string|max:15',
        'device_type' => 'required|string', // Add validation for device_type
        'laptop_type' => 'required|string',
        'company_name' => 'nullable|string|max:255',
        'assign_date' => 'nullable|date',
        'microsoft_account' => 'nullable|string|max:255',
        'password' => 'nullable|string|max:255',
        'recovery_code' => 'nullable|string|max:255',
        'serial_number' => 'required|string|max:255|unique:laptops,serial_number',
        'host_info' => 'nullable|string',
        'ram' => 'required|string',
        'storage' => 'required|string',
        'bill' => 'nullable|mimes:png,jpg,jpeg,pdf|max:2048'
    ]);

    $employee = Employee::findOrFail($request->employee_id);

    $laptop = Laptop::create([
        'serial_number' => $request->serial_number,
        'device_type' => $request->device_type, // Storing device_type in database
        'laptop_type' => $request->laptop_type,
        'company_name' => $request->company_name,
        'assign_date' => $request->assign_date,
        'microsoft_account' => $request->microsoft_account,
        'password' => $request->password,
        'recovery_code' => $request->recovery_code,
        'employee_id' => $employee->id,
        'is_personal' => $request->laptop_type === 'personal',
        'host_info' => $request->host_info,
        'ram' => $request->ram,
        'storage' => $request->storage,
    ]);

    LaptopHistory::create([
        'laptop_id' => $laptop->id,
        'employee_id' => $employee->id,
        'action' => 'Assigned',
        'date' => $request->assign_date,
        'notes' => 'Laptop assigned to employee.',
    ]);

    if ($request->hasFile('bill')) {
        $file = $request->file('bill');
        $filePath = $file->store('bills', 'public'); 
    }

    return redirect()->route('laptop_tracker.index')->with('success', 'Laptop assigned successfully!');
}

    public function edit($id)
    {
        $laptop = Laptop::findOrFail($id);
        $employees = Employee::orderBy('id', 'desc')->get();
        return view('pages.laptop_tracker.edit', compact('laptop', 'employees'));
    }

    public function update(Request $request, $id)
{
    $request->validate([
        'employee_id' => 'required|exists:employees,id',
        'serial_number' => 'required|string|max:255|unique:laptops,serial_number,' . $id,
        'laptop_type' => 'required|string',
        'company_name' => 'nullable|string|max:255',
        'assign_date' => 'nullable|date',
        'microsoft_account' => 'nullable|string|max:255',
        'password' => 'nullable|string|max:255',
        'recovery_code' => 'nullable|string|max:255',
        'host_info' => 'nullable|string',
        'notes' => 'nullable|string',
        'ram' => 'required|string',
        'storage' => 'required|string',
    ]);

    $laptop = Laptop::findOrFail($id);
    $oldEmployeeId = $laptop->employee_id;

    $laptop->update([
        'serial_number' => $request->serial_number,
        'laptop_type' => $request->laptop_type,
        'company_name' => $request->company_name,
        'assign_date' => $request->assign_date,
        'microsoft_account' => $request->microsoft_account,
        'password' => $request->password,
        'recovery_code' => $request->recovery_code,
        'employee_id' => $request->employee_id,
        'is_personal' => $request->laptop_type === 'personal',
        'host_info' => $request->host_info,
        'ram' => $request->ram,
        'storage' => $request->storage,
    ]);

    if ($oldEmployeeId != $request->employee_id) {
        LaptopHistory::create([
            'laptop_id' => $laptop->id,
            'employee_id' => $request->employee_id,
            'action' => 'Reassigned',
            'date' => now(),
            'notes' => $request->notes,
        ]);
    }

    return redirect()->route('laptop_tracker.index')->with('success', 'Laptop updated successfully!');
}


public function reassign(Request $request, $id)
{
    $request->validate([
        'employee_id' => 'required|exists:employees,id',
        'notes' => 'nullable|string|max:255',
    ]);

    $laptop = Laptop::findOrFail($id);
    $oldEmployeeId = $laptop->employee_id;

    $laptop->update([
        'employee_id' => $request->employee_id,
    ]);

    LaptopHistory::create([
        'laptop_id' => $laptop->id,
        'employee_id' => $request->employee_id,
        'action' => 'Reassigned',
        'date' => now(),
        'notes' => $request->notes ?: 'Laptop reassigned from employee ID ' . $oldEmployeeId . ' to ' . $request->employee_id,
    ]);

    return redirect()->route('laptop_tracker.index')->with('success', 'Laptop reassigned successfully!');
}

    public function destroy($id)
    {
        $laptop = Laptop::findOrFail($id);
        $laptop->delete();

        return redirect()->route('laptop_tracker.index')->with('success', 'Laptop deleted successfully!');
    }

    public function history($id)
    {
        $laptop = Laptop::with('history.employee')->findOrFail($id);
        return view('pages.laptop_tracker.history', compact('laptop'));
    }

    public function repair(Request $request, $id)
{
    $request->validate([
        'repair_details' => 'required|string|max:255',
        'repaired_by' => 'required|string|max:255',
        'cost' => 'required|numeric',
        'approved_by' => 'required|string|max:255',
        'bill' => 'nullable|mimes:png,jpg,jpeg,pdf|max:2048', 
    ]);

 
    $laptop = Laptop::findOrFail($id);

    // Handle file upload
    $billPath = null;
    if ($request->hasFile('bill')) {
        $file = $request->file('bill');
        $billPath = $file->store('bills', 'public');
    }

    $laptop->repairs()->create([
        'repair_details' => $request->repair_details,
        'repaired_by' => $request->repaired_by,
        'cost' => $request->cost,
        'approved_by' => $request->approved_by,
        'bill' => $billPath, // Store file path
        'created_at' => now(),
    ]);
     LaptopHistory::create([
        'laptop_id' => $laptop->id,
        'employee_id' => $laptop->employee_id,
        'action' => 'Repaired',
        'date' => now(),
        'notes' => 'Laptop repaired by ' . $request->repaired_by . ' at a cost of ' . $request->cost,
    ]);

    return redirect()->route('laptop_tracker.index')->with('success', 'Laptop repair details added successfully!');
}


public function editRepairModel($id)
{
    $repair = LaptopRepair::findOrFail($id); // Assuming you use LaptopRepair as the model
    return view('repairs.edit', compact('repair'));
}

public function updateRepairModel(Request $request, $id)
{
    $request->validate([
        'repair_details' => 'required|string|max:255',
        'repaired_by' => 'required|string|max:255',
        'cost' => 'required|numeric',
        'approved_by' => 'required|string|max:255',
        'bill' => 'nullable|mimes:png,jpg,jpeg,pdf|max:2048', 
    ]);

    $repair = LaptopRepair::findOrFail($id);

    // Handle file upload
    $billPath = $repair->bill; // Keep the old file if no new file is uploaded
    if ($request->hasFile('bill')) {
        // Delete the old bill file if it exists
        if ($repair->bill && Storage::disk('public')->exists($repair->bill)) {
            Storage::disk('public')->delete($repair->bill);
        }
        // Store the new bill file
        $billPath = $request->file('bill')->store('bills', 'public');
    }

    // Update repair details
    $repair->update([
        'repair_details' => $request->repair_details,
        'repaired_by' => $request->repaired_by,
        'cost' => $request->cost,
        'approved_by' => $request->approved_by,
        'bill' => $billPath,
    ]);

    return redirect()->route('laptop_tracker.index')->with('success', 'Repair details updated successfully.');
}

public function destroyRepairModel($id)
{
    $repair = LaptopRepair::findOrFail($id);

    // Delete the bill file if it exists
    if ($repair->bill && Storage::disk('public')->exists($repair->bill)) {
        Storage::disk('public')->delete($repair->bill);
    }

    // Delete the corresponding history entry for this repair
    $history = LaptopHistory::where('laptop_id', $repair->laptop_id)
                            ->where('action', 'Repaired')
                            ->where('notes', 'LIKE', '%Laptop repaired by ' . $repair->repaired_by . '%')
                            ->first();

    if ($history) {
        $history->delete();
    }

    // Delete the repair record
    $repair->delete();

    return redirect()->route('laptop_tracker.index')->with('success', 'Repair deleted successfully.');
}





    public function activeLaptopsByDepartment($department)
    {
         $activeLaptops = Laptop::whereHas('employee', function ($query) use ($department) {
            $query->where('department', $department);
        })->with('employee')->paginate(10); 
        $employees = Employee::where('department', $department)->get();

    
        return view('pages.laptop_tracker.active_laptops', compact('activeLaptops', 'department','employees'));
    }
    


    public function showDevicesByDepartment(Request $request, $department)
    {
        // Query to get devices (laptops/desktops) assigned to this department
        $query = Laptop::with('employee')
                    ->whereHas('employee', function($query) use ($department) {
                        $query->where('department', $department);
                    });
    
        // Search functionality
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('serial_number', 'like', "%{$search}%")
                  ->orWhereHas('employee', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }
    
        $devices = $query->paginate(10);
    
        // Get total laptops and desktops
        $totalLaptops = Laptop::whereHas('employee', function($q) use ($department) {
                                $q->where('department', $department);
                            })
                            ->where('device_type', 'laptop')
                            ->count();
    
        $totalDesktops = Laptop::whereHas('employee', function($q) use ($department) {
                                $q->where('department', $department);
                            })
                            ->where('device_type', 'desktop')
                            ->count();
    
        return view('pages.laptop_tracker.department_devices', compact('devices', 'department', 'totalLaptops', 'totalDesktops'));
    }
    
    
}
