<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\FollowUp;
use Illuminate\Support\Facades\Http;
use Twilio\Rest\Client;

class FollowUpController extends Controller
{
    public function index(Request $request)
    {
        $search = $request->input('search');
        
        // Query to fetch follow-ups, optionally filter based on $search
        $followupPt = FollowUp::where(function ($query) use ($search) {
                    $query->where('name', 'like', '%'.$search.'%')
                          ->orWhere('phone', 'like', '%'.$search.'%')
                          ->orWhere('doctor', 'like', '%'.$search.'%');
                })
                ->orderBy('created_at', 'desc')
                ->get();
    
        return view('pages.followup.index', compact('followupPt'));
    }
    



    public function create(){
        return view('pages.followup.create');
    }

   

    public function store(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'phone' => 'required|string|max:255',
        'doctor' => 'required|string|max:255',
        'department' => 'required|string|max:255',
        'status' => 'nullable|string|max:255',
        'remark' => 'required|string',
    ]);

    // Create the follow-up record
    $followUp = FollowUp::create([
        'name' => $request->input('name'),
        'phone' => $request->input('phone'),
        'doctor' => $request->input('doctor'),
        'department' => $request->input('department'),
        'status' => $request->input('status'),
        'remark' => $request->input('remark'),
    ]);

    // Format the phone number to E.164 format
    $phoneNumber = $this->formatPhoneNumber($followUp->phone);

    // Send WhatsApp message using Twilio
    $sid    = config('services.twilio.sid');
    $token  = config('services.twilio.token');
    $twilioWhatsAppNumber = config('services.twilio.whatsapp_number');
    $twilioSMSNumber = config('services.twilio.phone_number');
    $twilio = new Client($sid, $token);

    // Message body with clickable links
    $messageBody = "Dear {$followUp->name},\n\nThank you for reaching out to Madalasa Hospital. We have received your inquiry regarding your consultation with Dr. {$followUp->doctor} in the {$followUp->department} department. We appreciate your interest and will ensure that your follow-up details are handled promptly.\n\nWant to book an appointment? Click here: https://madalasahospital.com/book-appointment\n\nVisit our website: https://madalasahospital.com\n\nCall us now: +919319661010\n\nBest regards,\nMadalasa Hospital Team";

    // Check if the phone number is verified
    $verifiedNumbers = explode(',', env('TWILIO_VERIFIED_NUMBERS'));
    
    try {
        if (in_array($phoneNumber, $verifiedNumbers)) {
            // Attempt to send WhatsApp message
            $twilio->messages->create(
                "whatsapp:" . $phoneNumber, // to
                [
                    "from" => "whatsapp:" . $twilioWhatsAppNumber,
                    "body" => $messageBody
                ]
            );

            // Send SMS
            $twilio->messages->create(
                $phoneNumber, // to
                [
                    "from" => $twilioSMSNumber,
                    "body" => $messageBody
                ]
            );
        } else {
            // Send SMS if the number is not verified
            $twilio->messages->create(
                $phoneNumber, // to
                [
                    "from" => $twilioSMSNumber,
                    "body" => $messageBody
                ]
            );
        }
    } catch (\Twilio\Exceptions\RestException $e) {
        return redirect()->back()->with('error', 'Failed to send messages: ' . $e->getMessage());
    }

    return redirect()->back()->with('success', 'Follow-up record created and messages sent successfully.');
}

private function formatPhoneNumber($phone)
{
    // Assuming the phone number is an Indian number
    return '+91' . $phone;
}
 
}
