<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Employee;
use App\Models\Laptop;
use DB;


class AuthController extends Controller
{
    public function index(){
        return view('auth.login');
    }


    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'role' => 'required|in:superadmin,admin,camp,followup,laptop_tracker',
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $user = Auth::user();
            if ($user->role !== $request->role) {
                Auth::logout();
                return redirect()->back()->withErrors(['role' => 'Invalid role selected.']);
            }
            return $this->redirectToDashboard($user->role);
        }

        return redirect()->back()->withErrors(['email' => 'These credentials do not match our records.']);
    }

    protected function redirectToDashboard($role)
    {
        switch ($role) {
            case 'superadmin':
                return redirect()->route('superadmin.dashboard');
            case 'admin':
                return redirect()->route('admin.dashboard');
            case 'camp':
                return redirect()->route('office.dashboard');
            case 'followup':
                return redirect()->route('followup.dashboard');
            case 'laptop_tracker':
                return redirect()->route('laptop_tracker.dashboard');
            default:
                return redirect()->route('login');
        }
    }




    public function superadminDashboard()
    {
        return view('pages.superAdmin.dashboard');
    }

    public function adminDashboard()
    {
        return view('pages.admin.dashboard');
    }

    public function officeDashboard()
    {
        return view('pages.office.dashboard');
    }


    public function followupDashboard()
    {
        return view('pages.followup.dashboard');
    }
 
    public function laptopTrackerDashboard(Request $request)
{
    // Fetch all laptops with employees
    $query = Laptop::with(['employee', 'repairs']);
    
    if ($request->has('search')) {
        $search = $request->get('search');
        $query->where('serial_number', 'like', "%{$search}%")
              ->orWhereHas('employee', function ($q) use ($search) {
                  $q->where('name', 'like', "%{$search}%");
              });
    }
    
    if ($request->has('device_type')) {
        $deviceType = $request->get('device_type');
        $query->where('device_type', $deviceType);
    }

    // Retrieve paginated laptops
    $laptops = $query->paginate(10);

    // Fetch overall device counts
    $totalDevices = Laptop::count();
    $totalLaptops = Laptop::where('device_type', 'laptop')->count();
    $totalDesktops = Laptop::where('device_type', 'desktop')->count();
    $totalTablets = Laptop::where('device_type', 'tablet')->count();
    $totalMobiles = Laptop::where('device_type', 'mobile')->count();
    
    // Fetch employees without laptops
    $employeesWithoutLaptop = Employee::doesntHave('laptop')->count();
    
    // Fetch department-wise device counts
    $deviceCounts = DB::table('laptops')
        ->join('employees', 'laptops.employee_id', '=', 'employees.id')
        ->select(
            'employees.department',
            DB::raw('SUM(CASE WHEN device_type = "laptop" THEN 1 ELSE 0 END) as laptop_count'),
            DB::raw('SUM(CASE WHEN device_type = "desktop" THEN 1 ELSE 0 END) as desktop_count'),
            DB::raw('SUM(CASE WHEN device_type = "mobile" THEN 1 ELSE 0 END) as mobile_count')
        )
        ->groupBy('employees.department')
        ->get();

    // Return view with data
    return view('pages.laptop_tracker.dashboard', compact(
        'laptops', 
        'totalDevices', 
        'totalLaptops', 
        'totalDesktops', 
        'totalTablets', 
        'totalMobiles', 
        'employeesWithoutLaptop', 
        'deviceCounts'
    ));
}



    
    









    public function logout()
    {
        Auth::logout();
        return redirect()->route('login');
    }





}
